# MIT License
# Copyright (c) 2025 Nible Tecnología en Desarrollo LTDA
# See LICENSE file for more details.

from app import create_app
from core.utils.db import db
from flask_mail import Mail
from urllib.parse import urlencode
from core.utils.jinja_extends import setup_jinja2
from core.version import __version__ 
from project.version import __version__ as project_version

# Importaciones necesarias para el logging
import logging
from logging.handlers import RotatingFileHandler
import os

SYSTEM_VERSION = __version__
PROJECT_VERSION = project_version

# 1. Crear la instancia de la aplicación
app = create_app()
mail = Mail(app)

# --------------------------------------------------------------------
# 2. CONFIGURACIÓN DEL LOGGING PARA PRODUCCIÓN (¡NUEVO BLOQUE!)
# --------------------------------------------------------------------
if not app.debug:
    # Asegura que el directorio 'logs' exista
    if not os.path.exists('logs'):
        os.mkdir('logs')
        
    # Configura el manejador de archivos rotativo
    file_handler = RotatingFileHandler('logs/app.log', maxBytes=10240, backupCount=10)
    file_handler.setFormatter(logging.Formatter(
        '%(asctime)s %(levelname)s: %(message)s [in %(pathname)s:%(lineno)d]'
    ))
    file_handler.setLevel(logging.INFO) # Nivel de registro: INFO o superior
    
    # Añade el manejador de archivos al logger de Flask
    app.logger.addHandler(file_handler)
    
    # Opcional: Establecer el nivel del logger principal de Flask
    app.logger.setLevel(logging.INFO)
# --------------------------------------------------------------------

@app.context_processor
def inject_version():
    return {'version': SYSTEM_VERSION, 'project_version': PROJECT_VERSION} 

setup_jinja2(app)

with app.app_context():
    db.create_all()

if __name__ == "__main__":
    # Este bloque solo se ejecuta en modo de desarrollo local.
    # En producción (Gunicorn/uWSGI), el bloque 'if not app.debug:' es el que configura el logging.
    app.run(debug=True, host='0.0.0.0', port=5000, use_reloader=False)