# -*- coding: utf-8 -*-
from core.utils.db import db
from core.utils.packages.session import BackupOnDeleteMixin
from sqlalchemy.orm import backref
from sqlalchemy import BigInteger
from datetime import datetime, timezone 
from sqlalchemy import desc, func, String, case, literal_column  
from sqlalchemy.ext.hybrid import hybrid_property 
from sqlalchemy.sql.expression import null 

class Object(db.Model, BackupOnDeleteMixin):
    __tablename__ = 'object'
    id = db.Column(db.Integer, primary_key=True)
    creator_id = db.Column(db.Integer, db.ForeignKey('creator.id', ondelete='SET NULL'), nullable=True)
    creator = db.relationship('Creator', foreign_keys=[creator_id], backref=backref('objects', order_by=lambda: desc(Object.created_at),passive_deletes=True))
    coAuthors = db.Column(db.Text())
    title = db.Column(db.String(255))
    suggestedTitle = db.Column(db.Text())
    creationDate = db.Column(db.String(255))
    dateEra = db.Column(db.String(255))
    accessionNumber = db.Column(db.String(255))
    rfid = db.Column(db.String(255))
    collection_id = db.Column(db.Integer, db.ForeignKey('collectionlist.id', ondelete='SET NULL'), nullable=True)
    collection = db.relationship('Collectionlist', foreign_keys=[collection_id], backref=backref('objects', order_by=lambda: desc(Object.created_at),passive_deletes=True))
    lastLocation = db.Column(db.String(255))
    numObjects = db.Column(db.String(255))
    allMeasurements = db.Column(db.Text())
    object_category = db.Table( 'object_category', db.Column('object_id', db.Integer, db.ForeignKey('object.id'), primary_key=True), db.Column('category_id', db.Integer, db.ForeignKey('category.id'), primary_key=True) )
    categoryList = db.relationship('Category', secondary=object_category, lazy='subquery',backref=db.backref('objects', lazy='subquery') )
    object_type = db.Table( 'object_type', db.Column('object_id', db.Integer, db.ForeignKey('object.id'), primary_key=True), db.Column('type_id', db.Integer, db.ForeignKey('type.id'), primary_key=True) )
    objectTypeList = db.relationship('Type', secondary=object_type, lazy='subquery',backref=db.backref('objects', lazy='subquery') )
    object_technique = db.Table( 'object_technique', db.Column('object_id', db.Integer, db.ForeignKey('object.id'), primary_key=True), db.Column('technique_id', db.Integer, db.ForeignKey('technique.id'), primary_key=True) )
    methodsTechniquesList = db.relationship('Technique', secondary=object_technique, lazy='subquery',backref=db.backref('objects', lazy='subquery') )
    object_material = db.Table( 'object_material', db.Column('object_id', db.Integer, db.ForeignKey('object.id'), primary_key=True), db.Column('material_id', db.Integer, db.ForeignKey('material.id'), primary_key=True) )
    materialsList = db.relationship('Material', secondary=object_material, lazy='subquery',backref=db.backref('objects', lazy='subquery') )
    category = db.Column(db.String(255))
    objectType = db.Column(db.String(255))
    methodsTechniques = db.Column(db.String(255))
    material = db.Column(db.String(255))
    support = db.Column(db.String(255))
    description = db.Column(db.Text())
    signature = db.Column(db.Boolean())
    signatureLocation = db.Column(db.String(255))
    distinctiveFeatures = db.Column(db.Text())
    inscriptions = db.Column(db.Text())
    observations = db.Column(db.Text())
    process = db.Column(db.Text())
    iconographicDescription = db.Column(db.Text())
    references = db.Column(db.String(255))
    object_styleperiod = db.Table( 'object_styleperiod', db.Column('object_id', db.Integer, db.ForeignKey('object.id'), primary_key=True), db.Column('styleperiod_id', db.Integer, db.ForeignKey('styleperiod.id'), primary_key=True) )
    periods = db.relationship('Styleperiod', secondary=object_styleperiod, lazy='subquery',backref=db.backref('objects', lazy='subquery') )
    specify = db.Column(db.Text())
    approximateDate = db.Column(db.Boolean())
    noDate = db.Column(db.Boolean())
    object_cultureethnia = db.Table( 'object_cultureethnia', db.Column('object_id', db.Integer, db.ForeignKey('object.id'), primary_key=True), db.Column('cultureethnia_id', db.Integer, db.ForeignKey('cultureethnia.id'), primary_key=True) )
    culture = db.relationship('Cultureethnia', secondary=object_cultureethnia, lazy='subquery',backref=db.backref('objects', lazy='subquery') )
    cultureSpecify = db.Column(db.Text())
    useFunction = db.Column(db.Text())
    unknownOrigin = db.Column(db.Boolean())
    countryOrigin = db.Column(db.Text())
    stateProvince = db.Column(db.Text())
    locality = db.Column(db.String(255))
    specificPlace = db.Column(db.Text())
    object_customperiod = db.Table( 'object_customperiod', db.Column('object_id', db.Integer, db.ForeignKey('object.id'), primary_key=True), db.Column('customperiod_id', db.Integer, db.ForeignKey('customperiod.id'), primary_key=True) )
    thesaurusPeriod = db.relationship('Customperiod', secondary=object_customperiod, lazy='subquery',backref=db.backref('objects', lazy='subquery') )
    contextObservations = db.Column(db.Text())
    groupType = db.Column(db.String(255))
    groupName = db.Column(db.String(255))
    numComponents = db.Column(db.String(255))
    partOf = db.Column(db.String(255))
    principalAuthorRole = db.Column(db.Text())
    anonymous = db.Column(db.Boolean())
    unknownAuthor = db.Column(db.Boolean())
    attributedToCreator = db.Column(db.Boolean())
    conservationState = db.Column(db.Text())
    temperatureSpecifications = db.Column(db.Text())
    humidityVulnerable = db.Column(db.Boolean())
    humiditySpecifications = db.Column(db.Text())
    lightVulnerable = db.Column(db.Boolean())
    lightSpecifications = db.Column(db.Text())
    conservationRecommendations = db.Column(db.Text())
    ornamentalAlteration = db.Column(db.Boolean())
    ornamentalDeterioration = db.Column(db.Boolean())
    structuralAlteration = db.Column(db.Boolean())
    structuralDeterioration = db.Column(db.Boolean())
    otherDeterioration = db.Column(db.Boolean())
    conservationInformation = db.Column(db.Text())
    gpsLocation = db.Column(db.String(255))
    entryMethod = db.Column(db.Text())
    acquisitionDate = db.Column(db.String(255))
    acquisitionHistory = db.Column(db.Text())
    provenance = db.Column(db.Text())
    acquisitionCountry = db.Column(db.String(255))
    deaccessionDate = db.Column(db.String(255))
    deaccessionMethod = db.Column(db.Text())
    deaccessionObservations = db.Column(db.Text())
    currency = db.Column(db.Text())
    purchasePrice = db.Column(db.String(255))
    estimatedValue = db.Column(BigInteger(), nullable=True)
    salePrice = db.Column(db.String(255))
    appraisalDate = db.Column(db.String(255))
    appraiser = db.Column(db.Text())
    appraisalNotes = db.Column(db.Text())
    insuranceValue = db.Column(db.String(255))
    insurerName = db.Column(db.Text())
    policyNumber = db.Column(db.Text())
    costAdmin = db.Column(BigInteger(), nullable=True)
    costUSDAdmin = db.Column(BigInteger(), nullable=True)
    annualTDC_admin = db.Column(db.String(255))
    originAdmin = db.Column(db.Text())
    valueUSDAdmin = db.Column(BigInteger(), nullable=True)
    insuranceAmountAdmin = db.Column(BigInteger(), nullable=True)
    featuredImage_id = db.Column(db.Integer, db.ForeignKey('blob.id'), nullable=True)
    featuredImage = db.relationship('Blob', foreign_keys=[featuredImage_id], lazy='joined')
    object_objectgroup = db.Table( 'object_objectgroup', db.Column('object_id', db.Integer, db.ForeignKey('object.id'), primary_key=True), db.Column('objectgroup_id', db.Integer, db.ForeignKey('objectgroup.id'), primary_key=True) )
    objectGroup = db.relationship('Objectgroup', secondary=object_objectgroup, lazy='subquery',backref=db.backref('objects', lazy='subquery') )
    object_tag = db.Table( 'object_tag', db.Column('object_id', db.Integer, db.ForeignKey('object.id'), primary_key=True), db.Column('tag_id', db.Integer, db.ForeignKey('tag.id'), primary_key=True) )
    tags = db.relationship('Tag', secondary=object_tag, lazy='subquery',backref=db.backref('objects', lazy='subquery') )
    insuranceResponsible = db.Column(db.String(255))
    insuranceDateExpiration = db.Column(db.String(255))
    theme_id = db.Column(db.Integer, db.ForeignKey('theme.id', ondelete='SET NULL'), nullable=True)
    theme = db.relationship('Theme', foreign_keys=[theme_id], backref=backref('objects', order_by=lambda: desc(Object.created_at),passive_deletes=True))
    registrationStatus = db.Column(db.String(255))
    conservationStateSpecify = db.Column(db.String(255))
    ingressDate = db.Column(db.String(255))
    registerIndex = db.Column(db.Integer())
    typeAcquisition = db.Column(db.String(255))
    insurancePercent = db.Column(db.String(255))
    qtySD = db.Column(db.Integer())
    qtyHD = db.Column(db.Integer())
    qtyImage = db.Column(db.Integer())
    creationDateText = db.Column(db.String(255))
    additionalAccessionNumber = db.Column(db.String(255))
    createdby_id = db.Column(db.Integer, db.ForeignKey('user.id'))
    modifiedby_id = db.Column(db.Integer, db.ForeignKey('user.id'))
    createdby = db.relationship('User', backref='created_objects', foreign_keys=[createdby_id])
    modifiedby = db.relationship('User', backref='modified_objects', foreign_keys=[modifiedby_id])
    created_at = db.Column(db.DateTime, default=lambda: datetime.now(timezone.utc))
    updated_at = db.Column(db.DateTime, default=lambda: datetime.now(timezone.utc), onupdate=lambda: datetime.now(timezone.utc))

    def helper_lastLocation(self):
        from project.scripts.getFirstRecord import getLastLocation
        result = getLastLocation(self)
        self.lastLocation = result
        try:
            db.session.add(self) 
            db.session.commit() 
        except Exception as e:
            db.session.rollback()
            raise 
        return self.lastLocation
    def helper_featuredImage(self):
        from project.scripts.getFirstRecord import getFirstRecord
        result = getFirstRecord(self)
        self.featuredImage = result
        try:
            db.session.add(self) 
            db.session.commit() 
        except Exception as e:
            db.session.rollback()
            raise 
        return self.featuredImage
    def helper_registerIndex(self):
        from project.scripts.objectsQty import index
        result = index(self)
        self.registerIndex = result
        try:
            db.session.add(self) 
            db.session.commit() 
        except Exception as e:
            db.session.rollback()
            raise 
        return self.registerIndex
    def helper_qtySD(self):
        from project.scripts.stat import cantImagesSD
        result = cantImagesSD(self)
        self.qtySD = result
        try:
            db.session.add(self) 
            db.session.commit() 
        except Exception as e:
            db.session.rollback()
            raise 
        return self.qtySD
    def helper_qtyHD(self):
        from project.scripts.stat import cantImagesHD
        result = cantImagesHD(self)
        self.qtyHD = result
        try:
            db.session.add(self) 
            db.session.commit() 
        except Exception as e:
            db.session.rollback()
            raise 
        return self.qtyHD
    def helper_qtyImage(self):
        from project.scripts.stat import cantImages
        result = cantImages(self)
        self.qtyImage = result
        try:
            db.session.add(self) 
            db.session.commit() 
        except Exception as e:
            db.session.rollback()
            raise 
        return self.qtyImage

    @hybrid_property
    def full_repr(self):
        result = f'''{self.title if self.title is not None else ''} {self.suggestedTitle if self.suggestedTitle is not None else ''}'''
        return result.strip()

    @full_repr.expression
    def full_repr(cls):
        return (
            literal_column("") + literal_column("''")  + func.coalesce(cls.title, literal_column("''"))  + literal_column("' '") + literal_column("''")  + func.coalesce(cls.suggestedTitle, literal_column("''"))  + literal_column("''")
        )

    def __repr__(self) -> str:
        return self.full_repr 
    __default_ordering__ = [desc('created_at')]

    __table_args__ = {
        "mysql_charset": "utf8mb4",
        "mysql_collate": "utf8mb4_unicode_ci",
    }


def get_fields():
    fields = {'creator': {'id': 1, 'type': 'connected_table', 'name': 'creator', 'maxlength': None, 'connected_table': 14, 'label': 'Creador / Autor Principal', 'input': 'connected_table', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'coAuthors': {'id': 2, 'type': 'Text', 'name': 'coAuthors', 'maxlength': None, 'connected_table': None, 'label': 'Co-autores / Co-creadores', 'input': 'text', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': True, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'title': {'id': 3, 'type': 'String', 'name': 'title', 'maxlength': None, 'connected_table': None, 'label': 'Nombre / Título', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'suggestedTitle': {'id': 4, 'type': 'Text', 'name': 'suggestedTitle', 'maxlength': None, 'connected_table': None, 'label': 'Título Sugerido', 'input': 'text', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'creationDate': {'id': 5, 'type': 'String', 'name': 'creationDate', 'maxlength': None, 'connected_table': None, 'label': 'Fecha de Creación', 'input': 'incompletedate', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'dateEra': {'id': 6, 'type': 'String', 'name': 'dateEra', 'maxlength': None, 'connected_table': None, 'label': 'Fecha / Época', 'input': 'select', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '{\r\n"Primera época":"1",\r\n"Huacayñan":"2",\r\n"Edad de la Ira":"3",\r\n"Mientras vivo siempre te recuerdo":"4",\r\n"Varios":"5",\r\n"Capilla del Hombre":"6"}', 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'accessionNumber': {'id': 7, 'type': 'String', 'name': 'accessionNumber', 'maxlength': None, 'connected_table': None, 'label': 'No. de Acceso / Registro', 'input': 'text', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'rfid': {'id': 8, 'type': 'String', 'name': 'rfid', 'maxlength': None, 'connected_table': None, 'label': 'Código de Barras RFID', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'collection': {'id': 9, 'type': 'connected_table', 'name': 'collection', 'maxlength': None, 'connected_table': 13, 'label': 'Colección [lista]', 'input': 'connected_table', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'lastLocation': {'id': 11, 'type': 'String', 'name': 'lastLocation', 'maxlength': None, 'connected_table': None, 'label': 'Última Ubicación', 'input': 'text', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': True, 'calculate_file': 'getFirstRecord', 'calculate_function': 'getLastLocation', 'helper': 'script'}, 'numObjects': {'id': 12, 'type': 'String', 'name': 'numObjects', 'maxlength': None, 'connected_table': None, 'label': 'No. De Objetos / Serie', 'input': 'text', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': True, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'allMeasurements': {'id': 13, 'type': 'Text', 'name': 'allMeasurements', 'maxlength': None, 'connected_table': None, 'label': 'Todas Las Medidas', 'input': 'textarea', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': True, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'categoryList': {'id': 14, 'type': 'connected_table', 'name': 'categoryList', 'maxlength': None, 'connected_table': 8, 'label': 'Categoría [lista]', 'input': 'treeView', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': True, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'objectTypeList': {'id': 15, 'type': 'connected_table', 'name': 'objectTypeList', 'maxlength': None, 'connected_table': 2, 'label': 'Tipo de objeto [lista estándar]', 'input': 'treeView', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': True, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'methodsTechniquesList': {'id': 16, 'type': 'connected_table', 'name': 'methodsTechniquesList', 'maxlength': None, 'connected_table': 12, 'label': 'Métodos y Técnicas [lista estándar]', 'input': 'treeView', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': True, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'materialsList': {'id': 17, 'type': 'connected_table', 'name': 'materialsList', 'maxlength': None, 'connected_table': 10, 'label': 'Materiales [lista estándar]', 'input': 'treeView', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': True, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'category': {'id': 18, 'type': 'String', 'name': 'category', 'maxlength': None, 'connected_table': None, 'label': 'Categoría', 'input': 'text', 'sort': None, 'required': False, 'hidden': True, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': True, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'objectType': {'id': 19, 'type': 'String', 'name': 'objectType', 'maxlength': None, 'connected_table': None, 'label': 'Tipo de objeto', 'input': 'text', 'sort': 'None', 'required': False, 'hidden': True, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': True, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'methodsTechniques': {'id': 20, 'type': 'String', 'name': 'methodsTechniques', 'maxlength': None, 'connected_table': None, 'label': 'Métodos y Técnicas', 'input': 'text', 'sort': 'None', 'required': False, 'hidden': True, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': True, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'material': {'id': 21, 'type': 'String', 'name': 'material', 'maxlength': None, 'connected_table': None, 'label': 'Material', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'support': {'id': 22, 'type': 'String', 'name': 'support', 'maxlength': None, 'connected_table': None, 'label': 'Soporte', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'description': {'id': 23, 'type': 'Text', 'name': 'description', 'maxlength': None, 'connected_table': None, 'label': 'Descripción', 'input': 'textarea', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'signature': {'id': 24, 'type': 'Boolean', 'name': 'signature', 'maxlength': None, 'connected_table': None, 'label': 'Firma', 'input': 'boolean', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'signatureLocation': {'id': 25, 'type': 'String', 'name': 'signatureLocation', 'maxlength': None, 'connected_table': None, 'label': 'Ubicación / Texto de Firma', 'input': 'text', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'distinctiveFeatures': {'id': 26, 'type': 'Text', 'name': 'distinctiveFeatures', 'maxlength': None, 'connected_table': None, 'label': 'Características distintivas', 'input': 'textarea', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'inscriptions': {'id': 27, 'type': 'Text', 'name': 'inscriptions', 'maxlength': None, 'connected_table': None, 'label': 'Inscripciones y marcas', 'input': 'textarea', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'observations': {'id': 28, 'type': 'Text', 'name': 'observations', 'maxlength': None, 'connected_table': None, 'label': 'Observaciones', 'input': 'textarea', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'process': {'id': 29, 'type': 'Text', 'name': 'process', 'maxlength': None, 'connected_table': None, 'label': 'Proceso de Elaboración', 'input': 'textarea', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'iconographicDescription': {'id': 30, 'type': 'Text', 'name': 'iconographicDescription', 'maxlength': None, 'connected_table': None, 'label': 'Descripción iconográfica e iconológica', 'input': 'textarea', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'references': {'id': 31, 'type': 'String', 'name': 'references', 'maxlength': None, 'connected_table': None, 'label': 'Referencias / Tags / Metadatos', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'periods': {'id': 32, 'type': 'connected_table', 'name': 'periods', 'maxlength': None, 'connected_table': 11, 'label': 'Periodos Históricos / Artísticos / Arqueológicos', 'input': 'treeView', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': True, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'specify': {'id': 33, 'type': 'Text', 'name': 'specify', 'maxlength': None, 'connected_table': None, 'label': 'Especificar', 'input': 'text', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'approximateDate': {'id': 35, 'type': 'Boolean', 'name': 'approximateDate', 'maxlength': None, 'connected_table': None, 'label': 'Fecha Aproximada', 'input': 'boolean', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'noDate': {'id': 36, 'type': 'Boolean', 'name': 'noDate', 'maxlength': None, 'connected_table': None, 'label': 'Sin Fecha', 'input': 'boolean', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'culture': {'id': 37, 'type': 'connected_table', 'name': 'culture', 'maxlength': None, 'connected_table': 9, 'label': 'Cultura / Etnia', 'input': 'treeView', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': True, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'cultureSpecify': {'id': 38, 'type': 'Text', 'name': 'cultureSpecify', 'maxlength': None, 'connected_table': None, 'label': 'Cultura / Etnia (Especificar)', 'input': 'text', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'useFunction': {'id': 39, 'type': 'Text', 'name': 'useFunction', 'maxlength': None, 'connected_table': None, 'label': 'Uso / Función', 'input': 'textarea', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'unknownOrigin': {'id': 40, 'type': 'Boolean', 'name': 'unknownOrigin', 'maxlength': None, 'connected_table': None, 'label': 'Origen Desconocido', 'input': 'boolean', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'countryOrigin': {'id': 41, 'type': 'Text', 'name': 'countryOrigin', 'maxlength': None, 'connected_table': None, 'label': 'País de Origen', 'input': 'text', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'stateProvince': {'id': 42, 'type': 'Text', 'name': 'stateProvince', 'maxlength': None, 'connected_table': None, 'label': 'Estado / Provincia', 'input': 'text', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'locality': {'id': 43, 'type': 'String', 'name': 'locality', 'maxlength': None, 'connected_table': None, 'label': 'Localidad', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'specificPlace': {'id': 44, 'type': 'Text', 'name': 'specificPlace', 'maxlength': None, 'connected_table': None, 'label': 'Lugar Específico', 'input': 'textarea', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'thesaurusPeriod': {'id': 45, 'type': 'connected_table', 'name': 'thesaurusPeriod', 'maxlength': None, 'connected_table': 28, 'label': 'Período (Tesauro Institucional)', 'input': 'treeView', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': True, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'contextObservations': {'id': 46, 'type': 'Text', 'name': 'contextObservations', 'maxlength': None, 'connected_table': None, 'label': 'Observaciones de Contexto', 'input': 'textarea', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'groupType': {'id': 47, 'type': 'String', 'name': 'groupType', 'maxlength': None, 'connected_table': None, 'label': 'Tipo de Grupo', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'groupName': {'id': 48, 'type': 'String', 'name': 'groupName', 'maxlength': None, 'connected_table': None, 'label': 'Nombre del grupo', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'numComponents': {'id': 49, 'type': 'String', 'name': 'numComponents', 'maxlength': None, 'connected_table': None, 'label': 'No. de Componentes', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'partOf': {'id': 50, 'type': 'String', 'name': 'partOf', 'maxlength': None, 'connected_table': None, 'label': 'Forma parte de', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'principalAuthorRole': {'id': 51, 'type': 'Text', 'name': 'principalAuthorRole', 'maxlength': None, 'connected_table': None, 'label': 'Rol del Autor Principal', 'input': 'select', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'anonymous': {'id': 52, 'type': 'Boolean', 'name': 'anonymous', 'maxlength': None, 'connected_table': None, 'label': 'Anónimo', 'input': 'boolean', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'unknownAuthor': {'id': 53, 'type': 'Boolean', 'name': 'unknownAuthor', 'maxlength': None, 'connected_table': None, 'label': 'Desconocido', 'input': 'boolean', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'attributedToCreator': {'id': 54, 'type': 'Boolean', 'name': 'attributedToCreator', 'maxlength': None, 'connected_table': None, 'label': 'Es atribuido al creador', 'input': 'boolean', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'conservationState': {'id': 55, 'type': 'Text', 'name': 'conservationState', 'maxlength': None, 'connected_table': None, 'label': 'Estado de Conservación', 'input': 'select', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '{\r\n    "Excelente": "1",\r\n    "Muy Bueno": "2",\r\n    "Bueno": "3",\r\n    "Deficiente": "4",\r\n    "Malo": "5"\r\n}', 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'temperatureSpecifications': {'id': 56, 'type': 'Text', 'name': 'temperatureSpecifications', 'maxlength': None, 'connected_table': None, 'label': 'Especificaciones de Temperatura', 'input': 'text', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'humidityVulnerable': {'id': 57, 'type': 'Boolean', 'name': 'humidityVulnerable', 'maxlength': None, 'connected_table': None, 'label': 'Vulnerable a la Humedad', 'input': 'boolean', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'humiditySpecifications': {'id': 58, 'type': 'Text', 'name': 'humiditySpecifications', 'maxlength': None, 'connected_table': None, 'label': 'Especificaciones de Humedad', 'input': 'text', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'lightVulnerable': {'id': 59, 'type': 'Boolean', 'name': 'lightVulnerable', 'maxlength': None, 'connected_table': None, 'label': 'Vulnerable a la luz', 'input': 'boolean', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'lightSpecifications': {'id': 60, 'type': 'Text', 'name': 'lightSpecifications', 'maxlength': None, 'connected_table': None, 'label': 'Especificaciones de Luz', 'input': 'text', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'conservationRecommendations': {'id': 61, 'type': 'Text', 'name': 'conservationRecommendations', 'maxlength': None, 'connected_table': None, 'label': 'Recomendaciones de conservación', 'input': 'textarea', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'ornamentalAlteration': {'id': 62, 'type': 'Boolean', 'name': 'ornamentalAlteration', 'maxlength': None, 'connected_table': None, 'label': 'Alteración Ornamental', 'input': 'boolean', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'ornamentalDeterioration': {'id': 63, 'type': 'Boolean', 'name': 'ornamentalDeterioration', 'maxlength': None, 'connected_table': None, 'label': 'Deterioro ornamental', 'input': 'boolean', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'structuralAlteration': {'id': 64, 'type': 'Boolean', 'name': 'structuralAlteration', 'maxlength': None, 'connected_table': None, 'label': 'Alteración estructural', 'input': 'boolean', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'structuralDeterioration': {'id': 65, 'type': 'Boolean', 'name': 'structuralDeterioration', 'maxlength': None, 'connected_table': None, 'label': 'Deterioro estructural', 'input': 'boolean', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'otherDeterioration': {'id': 66, 'type': 'Boolean', 'name': 'otherDeterioration', 'maxlength': None, 'connected_table': None, 'label': 'Otro deterioro', 'input': 'boolean', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'conservationInformation': {'id': 67, 'type': 'Text', 'name': 'conservationInformation', 'maxlength': None, 'connected_table': None, 'label': 'Información de Conservación Detallada', 'input': 'textarea', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'gpsLocation': {'id': 68, 'type': 'String', 'name': 'gpsLocation', 'maxlength': None, 'connected_table': None, 'label': 'Localización GPS', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'entryMethod': {'id': 70, 'type': 'Text', 'name': 'entryMethod', 'maxlength': None, 'connected_table': None, 'label': 'Forma de Ingreso', 'input': 'select', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '{\r\n    "Comodato": "r",\r\n    "Compra": "c",\r\n    "Confiscación / Decomiso": "d",\r\n    "Creación propia": "a",\r\n    "Donación": "e",\r\n    "Encontrado": "i",\r\n    "Excavación": "q",\r\n    "Herencia": "j",\r\n    "Intercambio": "h",\r\n    "Préstamo": "b",\r\n    "Préstamo - Condicional": "m",\r\n    "Préstamo - Permanente": "k",\r\n    "Préstamo - Temporal": "l",\r\n    "Recolección de Campo": "f",\r\n    "Regalo": "p",\r\n    "Regalo con usufructo": "o"\r\n}', 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'acquisitionDate': {'id': 71, 'type': 'String', 'name': 'acquisitionDate', 'maxlength': None, 'connected_table': None, 'label': 'Fecha de Adquisición', 'input': 'incompletedate', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'acquisitionHistory': {'id': 72, 'type': 'Text', 'name': 'acquisitionHistory', 'maxlength': None, 'connected_table': None, 'label': 'Historial de Adquisicion', 'input': 'textarea', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'provenance': {'id': 73, 'type': 'Text', 'name': 'provenance', 'maxlength': None, 'connected_table': None, 'label': 'Proveniencia', 'input': 'textarea', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'acquisitionCountry': {'id': 74, 'type': 'String', 'name': 'acquisitionCountry', 'maxlength': None, 'connected_table': None, 'label': 'País de Adquisición', 'input': 'select', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '{\r\n    "Afganistán": "AF",\r\n    "Albania": "AL",\r\n    "Alemania": "DE",\r\n    "Andorra": "AD",\r\n    "Angola": "AO",\r\n    "Anguila": "AI",\r\n    "Antártida": "AQ",\r\n    "Antigua y Barbuda": "AG",\r\n    "Arabia Saudita": "SA",\r\n    "Argelia": "DZ",\r\n    "Argentina": "AR",\r\n    "Armenia": "AM",\r\n    "Aruba": "AW",\r\n    "Australia": "AU",\r\n    "Austria": "AT",\r\n    "Azerbaiyán": "AZ",\r\n    "Bahamas": "BS",\r\n    "Bahréin": "BH",\r\n    "Bangladés": "BD",\r\n    "Barbados": "BB",\r\n    "Bélgica": "BE",\r\n    "Belice": "BZ",\r\n    "Benín": "BJ",\r\n    "Bermudas": "BM",\r\n    "Bielorrusia": "BY",\r\n    "Birmania (Myanmar)": "MM",\r\n    "Bolivia": "BO",\r\n    "Bonaire, San Eustaquio y Saba": "BQ",\r\n    "Bosnia y Herzegovina": "BA",\r\n    "Botsuana": "BW",\r\n    "Brasil": "BR",\r\n    "Brunéi": "BN",\r\n    "Bulgaria": "BG",\r\n    "Burkina Faso": "BF",\r\n    "Burundi": "BI",\r\n    "Bután": "BT",\r\n    "Cabo Verde": "CV",\r\n    "Camboya": "KH",\r\n    "Camerún": "CM",\r\n    "Canadá": "CA",\r\n    "Catar": "QA",\r\n    "Chad": "TD",\r\n    "Chile": "CL",\r\n    "China": "CN",\r\n    "Chipre": "CY",\r\n    "Ciudad del Vaticano": "VA",\r\n    "Colombia": "CO",\r\n    "Comoras": "KM",\r\n    "Congo (Brazzaville)": "CG",\r\n    "Congo (Kinshasa)": "CD",\r\n    "Corea del Norte": "KP",\r\n    "Corea del Sur": "KR",\r\n    "Costa de Marfil": "CI",\r\n    "Costa Rica": "CR",\r\n    "Croacia": "HR",\r\n    "Cuba": "CU",\r\n    "Curazao": "CW",\r\n    "Dinamarca": "DK",\r\n    "Dominica": "DM",\r\n    "Ecuador": "EC",\r\n    "Egipto": "EG",\r\n    "El Salvador": "SV",\r\n    "Emiratos Árabes Unidos": "AE",\r\n    "Eritrea": "ER",\r\n    "Eslovaquia": "SK",\r\n    "Eslovenia": "SI",\r\n    "España": "ES",\r\n    "Estados Unidos": "US",\r\n    "Estonia": "EE",\r\n    "Esuatini": "SZ",\r\n    "Etiopía": "ET",\r\n    "Filipinas": "PH",\r\n    "Finlandia": "FI",\r\n    "Fiyi": "FJ",\r\n    "Francia": "FR",\r\n    "Gabón": "GA",\r\n    "Gambia": "GM",\r\n    "Georgia": "GE",\r\n    "Ghana": "GH",\r\n    "Gibraltar": "GI",\r\n    "Granada": "GD",\r\n    "Grecia": "GR",\r\n    "Groenlandia": "GL",\r\n    "Guadalupe": "GP",\r\n    "Guam": "GU",\r\n    "Guatemala": "GT",\r\n    "Guayana Francesa": "GF",\r\n    "Guernsey": "GG",\r\n    "Guinea": "GN",\r\n    "Guinea Ecuatorial": "GQ",\r\n    "Guinea-Bisáu": "GW",\r\n    "Guyana": "GY",\r\n    "Haití": "HT",\r\n    "Honduras": "HN",\r\n    "Hong Kong": "HK",\r\n    "Hungría": "HU",\r\n    "India": "IN",\r\n    "Indonesia": "ID",\r\n    "Irán": "IR",\r\n    "Irak": "IQ",\r\n    "Irlanda": "IE",\r\n    "Islandia": "IS",\r\n    "Isla Bouvet": "BV",\r\n    "Isla de Man": "IM",\r\n    "Isla de Navidad": "CX",\r\n    "Isla Norfolk": "NF",\r\n    "Islas Caimán": "KY",\r\n    "Islas Cocos": "CC",\r\n    "Islas Cook": "CK",\r\n    "Islas Feroe": "FO",\r\n    "Islas Georgia del Sur y Sandwich del Sur": "GS",\r\n    "Islas Heard y McDonald": "HM",\r\n    "Islas Malvinas": "FK",\r\n    "Islas Marianas del Norte": "MP",\r\n    "Islas Marshall": "MH",\r\n    "Islas Salomón": "SB",\r\n    "Islas Turcas y Caicos": "TC",\r\n    "Islas Ultramarinas Menores de Estados Unidos": "UM",\r\n    "Islas Vírgenes Británicas": "VG",\r\n    "Islas Vírgenes de los Estados Unidos": "VI",\r\n    "Israel": "IL",\r\n    "Italia": "IT",\r\n    "Jamaica": "JM",\r\n    "Japón": "JP",\r\n    "Jersey": "JE",\r\n    "Jordania": "JO",\r\n    "Kazajistán": "KZ",\r\n    "Kenia": "KE",\r\n    "Kirguistán": "KG",\r\n    "Kiribati": "KI",\r\n    "Kuwait": "KW",\r\n    "Laos": "LA",\r\n    "Lesoto": "LS",\r\n    "Letonia": "LV",\r\n    "Líbano": "LB",\r\n    "Liberia": "LR",\r\n    "Libia": "LY",\r\n    "Liechtenstein": "LI",\r\n    "Lituania": "LT",\r\n    "Luxemburgo": "LU",\r\n    "Macao": "MO",\r\n    "Macedonia del Norte": "MK",\r\n    "Madagascar": "MG",\r\n    "Malasia": "MY",\r\n    "Malaui": "MW",\r\n    "Maldivas": "MV",\r\n    "Malí": "ML",\r\n    "Malta": "MT",\r\n    "Marruecos": "MA",\r\n    "Martinica": "MQ",\r\n    "Mauricio": "MU",\r\n    "Mauritania": "MR",\r\n    "Mayotte": "YT",\r\n    "México": "MX",\r\n    "Micronesia": "FM",\r\n    "Moldavia": "MD",\r\n    "Mónaco": "MC",\r\n    "Mongolia": "MN",\r\n    "Montenegro": "ME",\r\n    "Montserrat": "MS",\r\n    "Mozambique": "MZ",\r\n    "Namibia": "NA",\r\n    "Nauru": "NR",\r\n    "Nepal": "NP",\r\n    "Nicaragua": "NI",\r\n    "Níger": "NE",\r\n    "Nigeria": "NG",\r\n    "Niue": "NU",\r\n    "Noruega": "NO",\r\n    "Nueva Caledonia": "NC",\r\n    "Nueva Zelanda": "NZ",\r\n    "Omán": "OM",\r\n    "Países Bajos": "NL",\r\n    "Pakistán": "PK",\r\n    "Palaos": "PW",\r\n    "Palestina": "PS",\r\n    "Panamá": "PA",\r\n    "Papúa Nueva Guinea": "PG",\r\n    "Paraguay": "PY",\r\n    "Perú": "PE",\r\n    "Pitcairn": "PN",\r\n    "Polinesia Francesa": "PF",\r\n    "Polonia": "PL",\r\n    "Portugal": "PT",\r\n    "Puerto Rico": "PR",\r\n    "Reino Unido": "GB",\r\n    "República Centroafricana": "CF",\r\n    "República Checa": "CZ",\r\n    "República Dominicana": "DO",\r\n    "Reunión": "RE",\r\n    "Ruanda": "RW",\r\n    "Rumania": "RO",\r\n    "Rusia": "RU",\r\n    "Sáhara Occidental": "EH",\r\n    "Samoa": "WS",\r\n    "Samoa Americana": "AS",\r\n    "San Bartolomé": "BL",\r\n    "San Cristóbal y Nieves": "KN",\r\n    "San Martín (parte francesa)": "MF",\r\n    "San Martín (parte neerlandesa)": "SX",\r\n    "San Marino": "SM",\r\n    "San Pedro y Miquelón": "PM",\r\n    "San Vicente y las Granadas": "VC",\r\n    "Santa Helena, Ascensión y Tristán de Acuña": "SH",\r\n    "Santa Lucía": "LC",\r\n    "Santo Tomé y Príncipe": "ST",\r\n    "Senegal": "SN",\r\n    "Serbia": "RS",\r\n    "Seychelles": "SC",\r\n    "Sierra Leona": "SL",\r\n    "Singapur": "SG",\r\n    "Siria": "SY",\r\n    "Somalia": "SO",\r\n    "Sri Lanka": "LK",\r\n    "Sudáfrica": "ZA",\r\n    "Sudán": "SD",\r\n    "Sudán del Sur": "SS",\r\n    "Suecia": "SE",\r\n    "Suiza": "CH",\r\n    "Surinam": "SR",\r\n    "Svalbard y Jan Mayen": "SJ",\r\n    "Taiwán": "TW",\r\n    "Tailandia": "TH",\r\n    "Tanzania": "TZ",\r\n    "Tayikistán": "TJ",\r\n    "Territorio Británico del Océano Índico": "IO",\r\n    "Territorios Franceses del Sur": "TF",\r\n    "Timor Oriental": "TL",\r\n    "Togo": "TG",\r\n    "Tokelau": "TK",\r\n    "Tonga": "TO",\r\n    "Trinidad y Tobago": "TT",\r\n    "Túnez": "TN",\r\n    "Turkmenistán": "TM",\r\n    "Turquía": "TR",\r\n    "Tuvalu": "TV",\r\n    "Ucrania": "UA",\r\n    "Uganda": "UG",\r\n    "Uruguay": "UY",\r\n    "Uzbekistán": "UZ",\r\n    "Vanuatu": "VU",\r\n    "Venezuela": "VE",\r\n    "Vietnam": "VN",\r\n    "Wallis y Futuna": "WF",\r\n    "Yemen": "YE",\r\n    "Yibuti": "DJ",\r\n    "Zambia": "ZM",\r\n    "Zimbabue": "ZW"\r\n}', 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'deaccessionDate': {'id': 75, 'type': 'String', 'name': 'deaccessionDate', 'maxlength': None, 'connected_table': None, 'label': 'Fecha de Desaccesión', 'input': 'incompletedate', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'deaccessionMethod': {'id': 76, 'type': 'Text', 'name': 'deaccessionMethod', 'maxlength': None, 'connected_table': None, 'label': 'Forma de Desaccesión', 'input': 'select', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '{"Donación":"1",\r\n"Regalo":"2",\r\n"Venta":"3",\r\n"Desecho":"4",\r\n"Intercambio":"5"\r\n}', 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'deaccessionObservations': {'id': 77, 'type': 'Text', 'name': 'deaccessionObservations', 'maxlength': None, 'connected_table': None, 'label': 'Observaciones', 'input': 'textarea', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'currency': {'id': 78, 'type': 'Text', 'name': 'currency', 'maxlength': None, 'connected_table': None, 'label': 'Moneda', 'input': 'select', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '{\r\n    "USD$ Dólar Estadounidense": "2",\r\n    "MX$ Pesos Mexicanos": "5",\r\n    "EUR€ Euro": "1",\r\n    "Colones Costarricenses": "6"\r\n}', 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'purchasePrice': {'id': 79, 'type': 'String', 'name': 'purchasePrice', 'maxlength': None, 'connected_table': None, 'label': 'Precio de Compra', 'input': 'text', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'estimatedValue': {'id': 80, 'type': 'Money', 'name': 'estimatedValue', 'maxlength': None, 'connected_table': None, 'label': 'Valor Estimado', 'input': 'money', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'salePrice': {'id': 81, 'type': 'String', 'name': 'salePrice', 'maxlength': None, 'connected_table': None, 'label': 'Precio de venta', 'input': 'text', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'appraisalDate': {'id': 82, 'type': 'String', 'name': 'appraisalDate', 'maxlength': None, 'connected_table': None, 'label': 'Fecha de Avalúo', 'input': 'incompletedate', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'appraiser': {'id': 83, 'type': 'Text', 'name': 'appraiser', 'maxlength': None, 'connected_table': None, 'label': 'Evaluador', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'appraisalNotes': {'id': 84, 'type': 'Text', 'name': 'appraisalNotes', 'maxlength': None, 'connected_table': None, 'label': 'Notas de Avalúo', 'input': 'textarea', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'insuranceValue': {'id': 85, 'type': 'String', 'name': 'insuranceValue', 'maxlength': None, 'connected_table': None, 'label': 'Valor del Seguro', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'insurerName': {'id': 86, 'type': 'Text', 'name': 'insurerName', 'maxlength': None, 'connected_table': None, 'label': 'Nombre de Asegurador', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'policyNumber': {'id': 87, 'type': 'Text', 'name': 'policyNumber', 'maxlength': None, 'connected_table': None, 'label': 'No. De Póliza', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'costAdmin': {'id': 88, 'type': 'Money', 'name': 'costAdmin', 'maxlength': None, 'connected_table': None, 'label': 'Costo', 'input': 'money', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'costUSDAdmin': {'id': 89, 'type': 'Money', 'name': 'costUSDAdmin', 'maxlength': None, 'connected_table': None, 'label': 'Costo USD', 'input': 'money', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'annualTDC_admin': {'id': 90, 'type': 'String', 'name': 'annualTDC_admin', 'maxlength': None, 'connected_table': None, 'label': 'TDC anual promedio', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '', 'extraclass': '', 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'originAdmin': {'id': 91, 'type': 'Text', 'name': 'originAdmin', 'maxlength': None, 'connected_table': None, 'label': 'Origen', 'input': 'text', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'valueUSDAdmin': {'id': 92, 'type': 'Money', 'name': 'valueUSDAdmin', 'maxlength': None, 'connected_table': None, 'label': 'Valor en USD', 'input': 'money', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'insuranceAmountAdmin': {'id': 93, 'type': 'Money', 'name': 'insuranceAmountAdmin', 'maxlength': None, 'connected_table': None, 'label': 'Seguro en Valor', 'input': 'money', 'sort': 'None', 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'featuredImage': {'id': 125, 'type': 'blob', 'name': 'featuredImage', 'maxlength': None, 'connected_table': None, 'label': 'Imagen Destacada', 'input': 'image', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': True, 'calculate_file': 'getFirstRecord', 'calculate_function': 'getFirstRecord', 'helper': 'script'}, 'objectGroup': {'id': 245, 'type': 'connected_table', 'name': 'objectGroup', 'maxlength': None, 'connected_table': 21, 'label': 'Grupos / Exposiciones', 'input': 'tagsInput', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': True, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'tags': {'id': 249, 'type': 'connected_table', 'name': 'tags', 'maxlength': None, 'connected_table': 23, 'label': 'Etiquetas', 'input': 'tagsInput', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': True, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'insuranceResponsible': {'id': 250, 'type': 'String', 'name': 'insuranceResponsible', 'maxlength': None, 'connected_table': None, 'label': 'Responsable de la prima', 'input': 'select', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '{"Kaluz":"1",\r\n"GEK":"2",\r\n"BX+":"3",\r\n"Asturval":"4"}', 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'insuranceDateExpiration': {'id': 251, 'type': 'String', 'name': 'insuranceDateExpiration', 'maxlength': None, 'connected_table': None, 'label': 'Fecha de Expiración', 'input': 'incompletedate', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'theme': {'id': 253, 'type': 'connected_table', 'name': 'theme', 'maxlength': None, 'connected_table': 25, 'label': 'Temática', 'input': 'connected_table', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'registrationStatus': {'id': 254, 'type': 'String', 'name': 'registrationStatus', 'maxlength': None, 'connected_table': None, 'label': 'Estado de Registro', 'input': 'select', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': '{"En proceso":"a",\r\n"Completo":"b"}', 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'conservationStateSpecify': {'id': 255, 'type': 'String', 'name': 'conservationStateSpecify', 'maxlength': None, 'connected_table': None, 'label': 'Estado de Consevación (Específicar)', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'ingressDate': {'id': 256, 'type': 'String', 'name': 'ingressDate', 'maxlength': None, 'connected_table': None, 'label': 'Fecha de Ingreso', 'input': 'incompletedate', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'registerIndex': {'id': 257, 'type': 'Integer', 'name': 'registerIndex', 'maxlength': None, 'connected_table': None, 'label': 'Índice de Registro', 'input': 'number', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': True, 'calculate_file': 'objectsQty', 'calculate_function': 'index', 'helper': 'script'}, 'typeAcquisition': {'id': 258, 'type': 'String', 'name': 'typeAcquisition', 'maxlength': None, 'connected_table': None, 'label': 'Tipo de Adquisición', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'insurancePercent': {'id': 259, 'type': 'String', 'name': 'insurancePercent', 'maxlength': None, 'connected_table': None, 'label': 'Seguro en Porcentaje', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'qtySD': {'id': 260, 'type': 'Integer', 'name': 'qtySD', 'maxlength': None, 'connected_table': None, 'label': 'Imágenes SD', 'input': 'number', 'sort': None, 'required': False, 'hidden': True, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': True, 'calculate_file': 'stat', 'calculate_function': 'cantImagesSD', 'helper': 'script'}, 'qtyHD': {'id': 261, 'type': 'Integer', 'name': 'qtyHD', 'maxlength': None, 'connected_table': None, 'label': 'Imágenes HD', 'input': 'number', 'sort': None, 'required': False, 'hidden': True, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': True, 'calculate_file': 'stat', 'calculate_function': 'cantImagesHD', 'helper': 'script'}, 'qtyImage': {'id': 262, 'type': 'Integer', 'name': 'qtyImage', 'maxlength': None, 'connected_table': None, 'label': 'Cantidad de Imágenes', 'input': 'number', 'sort': None, 'required': False, 'hidden': True, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': True, 'calculate_file': 'stat', 'calculate_function': 'cantImages', 'helper': 'script'}, 'creationDateText': {'id': 338, 'type': 'String', 'name': 'creationDateText', 'maxlength': None, 'connected_table': None, 'label': 'Fecha de Creación (texto)', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}, 'additionalAccessionNumber': {'id': 339, 'type': 'String', 'name': 'additionalAccessionNumber', 'maxlength': None, 'connected_table': None, 'label': 'Número de Acceso (Adicional)', 'input': 'text', 'sort': None, 'required': False, 'hidden': False, 'publicBlob': False, 'default_value': None, 'select_options': None, 'extraclass': None, 'allowHtml': False, 'hasManyValues': False, 'readOnly': False, 'calculate_file': None, 'calculate_function': None, 'helper': None}}
    return fields


def get_relevants():
    relevants = {}
    return relevants
